globals 
[
  clock plot-clock tick-length     ;; clock variables
  max-tick-length                  ;; the largest a tick length is allowed to be
  avg-speed                        ;; average speed of the two particles
  total-energy                     ;; total energy of the two particles
  x-center y-center                ;; coordinates of center of mass
  done?                            ;; becomes true when one particles is about to 'leave' the graphics window
  after-collision?                 ;; for graphing purposes
]

breeds [ particles paths spinner ]

particles-own 
[
  speed mass energy                ;; particle variables
  last-collision
  my-path
]

paths-own                          ;; this breed is created to mark the particles' paths            
[
  my-particle
]


to setup
  ca
  set-default-shape particles "circle"
  set-default-shape spinner "clock"
  set-default-shape paths "line half"
  set clock 0
  set done? false
  set max-tick-length 0.1073
  set after-collision? false
  make-spinner
  make-particles
  update-variables
  do-plotting
end

to update-variables
  set x-center mean values-from particles [ xcor ]
  set y-center mean values-from particles [ ycor ]
  set avg-speed  mean values-from particles [speed]
  set total-energy sum values-from particles [energy]
end

to go
  ask particles [ move ]
  ask particles [ ask my-path [          
     set size 2 * distance myself
  ]]                                              ;;  resets the path as the particles are moving

  ask particles                                   ;;  each particle checks if it's on the same patch as the other
  [ ;; without-interruption is needed here so one collision is
    ;; happening at a time
    without-interruption
      [ check-for-collision]
  ]
  update-variables
  if show-center-of-mass?                         ;; marks a gray path along the particles' center of mass
  [ 
    ask patch (round x-center) (round y-center)
      [ set pcolor gray ]
  ]
  fade-patches

  calculate-tick-length
  set clock clock + tick-length
  do-plotting
  ask spinner                                      ;; shows the passage of time
  [
    set heading clock * 360
    set label floor clock
  ]
end


to go-mode

if run-mode = "one-collision"  [go-once stop]
if run-mode = "all-collision-angles" [all-collision-angles]
if run-mode = "all-reflection-angles" [all-reflection-angles]

end


to go-once                                          ;; a single collision
  setup                                            
  while [ not done? ]
  [ go
    ask particles
    [ if (abs round pxcor = screen-edge-x - 1) or (abs round pycor = screen-edge-y - 1)
      [ set done? true ]
    ]
  ]
end

to all-collision-angles                            ;; activated when the reflection angle is constant and the collision angle is varied
  go-once
  set collision-angle collision-angle + 15
  if collision-angle > 345
  [ set collision-angle 15 ]
end

to all-reflection-angles                           ;; activated when the collision angle is constant and the reflection angle is varied
  go-once
  set reflection-angle reflection-angle + 15
  if reflection-angle = 360
  [ set reflection-angle 0 ]
end

to calculate-tick-length
  ;; tick-length is calculated in such way that even the fastest
  ;; particle will jump at most 1 patch length in a clock tick. As
  ;; particles jump (speed * tick-length) at every clock tick, making
  ;; tick length the inverse of the speed of the fastest particle
  ;; (1/max speed) assures that. Having each particle advance at most
  ;; one patch-length is necessary for them not to "jump over" each
  ;; other without colliding.
  ifelse any? particles with [speed > 0]
    [ set tick-length min list (1 / (ceiling max values-from particles [speed])) max-tick-length ]
    [ set tick-length max-tick-length ]
end


to move  ;; particle procedure
  jump (speed * tick-length)
end

to check-for-collision  ;; particle procedure 
  if count other-particles-here = 1
  [
    ;; the following conditions are imposed on collision candidates:
    ;;   1. they must have a lower who number than my own, because collision
    ;;      code is asymmetrical: it must always happen from the point of view
    ;;      of just one particle.
    ;;   2. they must not be the same particle that we last collided with on
    ;;      this patch, so that we have a chance to leave the patch after we've
    ;;      collided with someone.
    let candidate random-one-of other-particles-here with
      [who < who-of myself and myself != last-collision]
    ;; we also only collide if one of us has non-zero speed. It's useless
    ;; (and incorrect, actually) for two particles with zero speed to collide.
    if (candidate != nobody) and (speed > 0 or speed-of candidate > 0)
    [
      collide-with candidate
      set last-collision candidate
      set last-collision-of candidate self
      set after-collision? true
    ]
  ]
end

;; implements a collision with another particle.
;;
;; THIS IS THE HEART OF THE PARTICLE SIMULATION, AND YOU ARE STRONGLY ADVISED
;; NOT TO CHANGE IT UNLESS YOU REALLY UNDERSTAND WHAT YOU'RE DOING!
;;
;; The two particles colliding are self and other-particle, and while the
;; collision is performed from the point of view of self, both particles are
;; modified to reflect its effects. This is somewhat complicated, so I'll
;; give a general outline here: 
;;   1. Do initial setup, and determine the heading between the reflected particles
;;      (call it theta).
;;   2. Convert the representation of the velocity of each particle from
;;      speed/heading to a theta-based vector whose first component is the
;;      particle's speed along theta, and whose second component is the speed
;;      perpendicular to theta.
;;   3. Modify the velocity vectors to reflect the effects of the collision.
;;      This involves:
;;        a. computing the velocity of the center of mass of the whole system
;;           along direction theta
;;        b. updating the along-theta components of the two velocity vectors.
;;   4. Convert from the theta-based vector representation of velocity back to
;;      the usual speed/heading representation for each particle.
;;   5. Perform final cleanup and update derived quantities.
to collide-with [ other-particle ] ;; particle procedure
  ;;; PHASE 1: initial setup

  ;; for convenience, grab some quantities from other-particle
  let mass2 mass-of other-particle
  let speed2 speed-of other-particle
  let heading2 heading-of other-particle
    
  ;; since particles are modeled as zero-size points, theta isn't meaningfully
  ;; defined. we can assign it randomly without affecting the model's outcome.
  let theta reflection-angle

  ;;; PHASE 2: convert velocities to theta-based vector representation

  ;; now convert my velocity from speed/heading representation to components
  ;; along theta and perpendicular to theta
  let v1t (speed * cos (theta - heading))
  let v1l (speed * sin (theta - heading))

  ;; do the same for other-particle
  let v2t (speed2 * cos (theta - heading2))
  let v2l (speed2 * sin (theta - heading2))



  ;;; PHASE 3: manipulate vectors to implement collision 

  ;; compute the velocity of the system's center of mass along theta
  let vcm (((mass * v1t) + (mass2 * v2t)) / (mass + mass2) )

  ;; now compute the new velocity for each particle along direction theta.
  ;; velocity perpendicular to theta is unaffected by a collision along theta,
  ;; so the next two lines actually implement the collision itself, in the
  ;; sense that the effects of the collision are exactly the following changes
  ;; in particle velocity.
  set v1t (2 * vcm - v1t)
  set v2t (2 * vcm - v2t)



  ;;; PHASE 4: convert back to normal speed/heading

  ;; now convert my velocity vector into my new speed and heading
  set speed sqrt ((v1t ^ 2) + (v1l ^ 2))
  set energy (0.5 * mass * speed ^ 2)
  ;; if the magnitude of the velocity vector is 0, atan is undefined. but
  ;; speed will be 0, so heading is irrelevant anyway. therefore, in that 
  ;; case we'll just leave it unmodified.
  if v1l != 0 or v1t != 0
    [ set heading (theta - (atan v1l v1t)) ]

  ;; and do the same for other-particle
  set speed-of other-particle sqrt ((v2t ^ 2) + (v2l ^ 2))
  set energy-of other-particle (0.5 * (mass-of other-particle) * (speed-of other-particle) ^ 2 )
  if v2l != 0 or v2t != 0
    [ set heading-of other-particle (theta - (atan v2l v2t)) ]



  ;; PHASE 5: final updates
  set-new-path
  ask other-particle
    [set-new-path]
  
end

to recolor  ;; particle procedure
  ifelse speed < (0.5 * 10)
  [
    set color blue + 2
  ]
  [
    ifelse speed > (1.5 * 10)
      [ set color red ]
      [ set color green ]
  ]
end

to fade-patches
  ask patches with [ pcolor != black ]
    [ set pcolor ( pcolor - 0.05 )
      if (round (pcolor mod 10) = black)
        [ set pcolor black ] ]
end


;;;
;;; drawing procedures
;;;


;; creates initial particles
to make-particles
  create-custom-particles 1 [
    set color pink
    set speed init-pink-speed
    set mass pink-mass
    set heading 180
    bk 2 * speed
  ]
  create-custom-particles 1 [
    set color blue
    set speed init-blue-speed
    set mass blue-mass
    set heading 180 + collision-angle
    bk 2 * speed
  ]
  ask particles
  [
    setup-particle
    set-new-path
  ]
  calculate-tick-length
end


to setup-particle  ;; particle procedure
  set size 2
  set energy (0.5 * mass * speed ^ 2 )
  set last-collision nobody
end

to set-new-path  ;; particle procedure
  hatch 1
  [
    set breed paths
    ;;the new path will be the my-path of the calling particle
    set my-path-of myself self
  ]
end

;;; plotting procedure
to do-plotting
  set-current-plot "Speeds"
  ifelse after-collision?
     [set plot-clock (clock - (tick-length))]
     [set plot-clock clock]

  ifelse speed-of turtle 1 = speed-of turtle 2
     [set-current-plot-pen "both"
      plotxy plot-clock (speed-of turtle 1)]

     [set-current-plot-pen "pink"
      plotxy plot-clock (speed-of turtle 1)
      set-current-plot-pen "blue"
      plotxy plot-clock (speed-of turtle 2)]
end

to make-spinner
  create-custom-spinner 1
  [
    setxy (screen-edge-x - 6) (screen-edge-y - 6) 
    set color grey - 1.5
    set size 10 
    set heading 0
    set label 0
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
399
10
772
404
60
60
3.0
1
10
1
1
1
0

CC-WINDOW
5
573
781
668
Command Center

MONITOR
85
323
198
372
average speed
avg-speed
1
1

MONITOR
200
323
313
372
total energy
total-energy
1
1

SLIDER
16
75
382
108
collision-angle
collision-angle
15.0
345.0
150.0
15.0
1
degrees

SLIDER
15
151
195
184
init-pink-speed
init-pink-speed
1.0
20.0
15.0
1.0
1
NIL

SLIDER
15
186
194
219
pink-mass
pink-mass
1.0
20.0
1.0
1.0
1
NIL

SLIDER
16
112
382
145
reflection-angle
reflection-angle
0.0
345.0
225.0
15.0
1
degrees

SLIDER
205
152
384
185
init-blue-speed
init-blue-speed
1.0
20.0
6.0
1.0
1
NIL

SLIDER
205
187
384
220
blue-mass
blue-mass
1.0
20.0
4.0
1.0
1
NIL

BUTTON
14
17
99
61
NIL
setup
NIL
1
T
OBSERVER
T
NIL

MONITOR
38
271
161
320
speed of pink
speed-of turtle 1  ;; turtle 0 is spinner
1
1

MONITOR
231
269
359
318
speed of blue
speed-of turtle 2
1
1

MONITOR
39
221
162
270
energy of pink
energy-of turtle 1
1
1

MONITOR
231
222
359
271
energy of blue
energy-of turtle 2
1
1

SWITCH
399
410
582
443
show-center-of-mass?
show-center-of-mass?
1
1
-1000

PLOT
30
376
366
559
Speeds
time
speed
0.0
5.0
0.0
20.0
true
true
PENS
"pink" 1.0 0 -65413 true
"blue" 1.0 0 -16776961 true
"both" 1.0 0 -16777216 true

BUTTON
299
18
381
63
go
go-mode
T
1
T
OBSERVER
T
NIL

CHOOSER
105
17
292
62
run-mode
run-mode
"one-collision" "all-collision-angles" "all-reflection-angles"
0

@#$#@#$#@
WHAT IS IT?
-----------

This is one in a series of GasLab models that use the same basic rules for what happens when particles run into each other. Each one has different features in order to show different aspects of the behavior of gases.  

This model is simplified to show the collision of only two particles, since this event is so hard to watch when there are many particles on the screen: given the initial motions of two colliding particles, what can we learn about their final motions from the principles of conservation of momentum and energy?

The model is related to one of the original CM StarLogo applications (under the name GPCEE) and is now ported to NetLogo as part of the PARTICIPATORY SIMULATIONS Project.


HOW IT WORKS 
------------ 

The particles are modeled as hard balls with no internal energy except that which is due to their motion.  Collisions between particles are elastic.  Particles are colored according to speed -- blue for slow, green for medium, and red for high speeds.

Coloring of the particles is with respect to one speed (10).  Particles with a speed less than 5 are blue, ones that are more than 15 are red, while all in those in-between are green.

Particles behave according to the following rules:
1. A particle moves in a straight line without changing its speed, unless it collides with another particle or bounces off the wall.  The particles are aimed to hit each other at the origin.
2. Two particles "collide" if they find themselves on the same patch (the Graphics Window is composed of a grid of small squares called patches).
3. A random axis is chosen, as if they are two balls that hit each other and this axis is the line connecting their centers.
4. They exchange momentum and energy along that axis, according to the conservation of momentum and energy.  This calculation is done in the center of mass system.
5. Each turtle is assigned its new velocity, energy, and heading.
6. If a turtle finds itself on or very close to a wall of the container, it "bounces" -- that is, reflects its direction and keeps its same speed.


HOW TO USE IT
-------------
Initial settings:
- COLLISION-ANGLE: Sets the angle that separates the pink and blue particles before the collision.
- REFLECTION-ANGLE: Sets the angle of the axis connecting the particles' centers when they collide with respect to the vertical axis. To calculate the outcome of the collision, the speeds of the two particles are projected onto this new axis and the new speeds and headings are computed. Other GasLab models use random values for "REFLECTION-ANGLE", but this model allows you to experiment with them one by one. This angle is called THETA in the code of the model.
- INIT-PINK-SPEED (or BLUE): Sets the initial speed of the pink (or blue) particle.
- PINK-MASS (or BLUE): Sets the mass of the pink (or blue) particle.

Other settings:
- SHOW-CENTER-OF-MASS?: If ON, the center of mass of the system will be shown in gray.

Buttons for running the model:
- SETUP
- RUN-MODE: Chooses between ONE COLLISION (just one run), ALL-COLLISION-ANGLES (loops through all the collision angles with 15-degrees steps) and ALL-REFLECTION-ANGLES(loops through all the reflection angles with 15-degrees steps).
- GO

Monitors:
- ENERGY OF PINK (or -BLUE): Shows the current energy of the pink (or blue) particle.
- SPEED OF PINK (or -BLUE): Shows the current speed of the pink (or blue) particle.
- AVERAGE SPEED: Shows the average of the speeds of the two particles.
- TOTAL ENERGY: Shows the sum of the energies of the two particles.

Plots:
- SPEEDS: speed of each of the particles over time.

RUNNING THE MODEL
-----------------
Set the reflection-angle to zero.  Draw a picture representing the two balls as they collide, with their two faces touching.  Make the line connecting their centers be the same as theta.  Draw vectors representing their motion. 

While running the following situations note the paths of the two particles.  Can you make sense of what they do?  Is it what you expected?

Choose a COLLISION-ANGLE and a REFLECTION-ANGLE and choose ONE-COLLISION to see one particular collision.

Choose a COLLISION-ANGLE and choose ALL-REFLECTION-ANGLES to cycle through all of the angles of reflection.  

Choose a REFLECTION-ANGLE and choose ALL-COLLISION-ANGLES to cycle through all of the angles of collision.


THINGS TO NOTICE
----------------
With COLLISION-ANGLE = 180 (directly across from each other) and REFLECTION-ANGLE = 90, it looks as if the two particles miss each other.  What is happening?

With REFLECTION-ANGLE = 45 degrees, the particles go off at right angles. Why?  Draw a picture of what is happening at the moment of collision. 

With REFLECTION-ANGLE = 0 degrees, the two particles reverse direction. Why?

What is the motion of the center of mass?  What would you expect it to be? 


THINGS TO TRY
-------------
Have the masses of the two particles be different.

Have the initial speeds of the two particles be different.

Change the initial positions and headings of the two particles.  As a simple case, set one on the y-axis and the other on the x-axis, (COLLISION-ANGLE = 90) each one heading toward the origin.  The center of mass is no longer stationary. Note its path.  Is it what you would expect?

If the center of mass is not stationary, the two particles often have different speeds after they collide, even when they have identical initial speeds and masses!  Why does this happen?  How can this satisfy the conservation of both energy and momentum?  

The fact that the velocities are not always the same after every kind of collision is essential to getting a distribution of velocities among identical particles after many collisions, which is what we observe with particles in a gas.  

Does this seem like a reasonable model for colliding particles? When is it reasonably valid, and when is it decidedly NOT valid?

When two particles collide, should theta be picked randomly -- each theta has an equal probability -- or in some other way?  Would this change the eventual velocity distribution among many particles?
 
After you have gotten used to observing and understanding these simple collisions, go to the "Free Gas" or "Gas in a Box" model.  Especially watch the particle whose path is traced in gray.  Does it make sense?  Can you picture each collision?

Record the velocities of each particle after each collision. After you have several sets of velocities, look at the entire velocity distribution. What do you notice? Is it the Maxwell-Boltzmann distribution?


CREDITS AND REFERENCES
----------------------
To refer to this model in academic publications, please use: Wilensky, U. (1998).  NetLogo GasLab Single Collision model. http://ccl.northwestern.edu/netlogo/models/GasLabSingleCollision. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

In other publications, please use: Copyright 1998 by Uri Wilensky.  All rights reserved.  See http://ccl.northwestern.edu/netlogo/models/GasLabSingleCollision for terms of use.

@#$#@#$#@
default
true
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7566196 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7566196 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7566196 true true 150 285 285 225 285 75 150 135
Polygon -7566196 true true 150 135 15 75 150 15 285 75
Polygon -7566196 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7566196 true true 96 182 108
Circle -7566196 true true 110 127 80
Circle -7566196 true true 110 75 80
Line -7566196 true 150 100 80 30
Line -7566196 true 150 100 220 30

butterfly
true
0
Polygon -7566196 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7566196 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7566196 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7566196 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7566196 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7566196 true true 47 195 58
Circle -7566196 true true 195 195 58

circle
false
0
Circle -7566196 true true 30 30 240

circle 2
false
0
Circle -7566196 true true 16 16 270
Circle -16777216 true false 46 46 210

clock
true
0
Circle -7566196 true true 30 30 240
Polygon -16777216 true false 150 31 128 75 143 75 143 150 158 150 158 75 173 75
Circle -16777216 true false 135 135 30

cow
false
0
Polygon -7566196 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7566196 true true 73 210 86 251 62 249 48 208
Polygon -7566196 true true 25 114 16 195 9 204 23 213 25 200 39 123

face happy
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7566196 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7566196 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7566196 true true 60 15 75 300
Polygon -7566196 true true 90 150 270 90 90 30
Line -7566196 true 75 135 90 135
Line -7566196 true 75 45 90 45

flower
false
0
Polygon -11352576 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7566196 true true 85 132 38
Circle -7566196 true true 130 147 38
Circle -7566196 true true 192 85 38
Circle -7566196 true true 85 40 38
Circle -7566196 true true 177 40 38
Circle -7566196 true true 177 132 38
Circle -7566196 true true 70 85 38
Circle -7566196 true true 130 25 38
Circle -7566196 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -11352576 true false 189 233 219 188 249 173 279 188 234 218
Polygon -11352576 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7566196 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7566196 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7566196 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7566196 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7566196 true 150 0 150 300

line half
true
0
Line -7566196 true 150 0 150 150

pentagon
false
0
Polygon -7566196 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7566196 true true 110 5 80
Polygon -7566196 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7566196 true true 127 79 172 94
Polygon -7566196 true true 195 90 240 150 225 180 165 105
Polygon -7566196 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7566196 true true 135 90 165 300
Polygon -7566196 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7566196 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7566196 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7566196 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7566196 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7566196 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7566196 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7566196 true true 30 30 270 270

square 2
false
0
Rectangle -7566196 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7566196 true true 60 270 150 0 240 270 15 105 285 105
Polygon -7566196 true true 75 120 105 210 195 210 225 120 150 75

target
false
0
Circle -7566196 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7566196 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7566196 true true 120 120 60

tree
false
0
Circle -7566196 true true 118 3 94
Rectangle -6524078 true false 120 195 180 300
Circle -7566196 true true 65 21 108
Circle -7566196 true true 116 41 127
Circle -7566196 true true 45 90 120
Circle -7566196 true true 104 74 152

triangle
false
0
Polygon -7566196 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7566196 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7566196 true true 4 45 195 187
Polygon -7566196 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7566196 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7566196 false true 24 174 42
Circle -7566196 false true 144 174 42
Circle -7566196 false true 234 174 42

turtle
true
0
Polygon -11352576 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -11352576 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -11352576 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -11352576 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -11352576 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7566196 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7566196 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7566196 true 150 285 150 15
Line -7566196 true 15 150 285 150
Circle -7566196 true true 120 120 60
Line -7566196 true 216 40 79 269
Line -7566196 true 40 84 269 221
Line -7566196 true 40 216 269 79
Line -7566196 true 84 40 221 269

x
false
0
Polygon -7566196 true true 270 75 225 30 30 225 75 270
Polygon -7566196 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 2.1beta4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
